#region References

using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using System.Collections;
using System.Text;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

using WORKLOAD_TABLE_NAMES = gov.va.med.vbecs.Common.DatabaseConstants.WorkloadTableNames;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;

#endregion

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>David Askew</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>03/10/2006</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// VistALink Access Layer class
	///</summary>

	#endregion

	public sealed class VistaCache
	{
		#region Constants

		private const string HCPCS_CODES_LOOKUP_RPC_NAME = "VBECS HCPCS CODES LOOKUP";
		private const string HOSPITAL_LOCATION_LOOKUP_RPC_NAME = "VBECS HOSPITAL LOCATION LOOKUP";
		private const string WORKLOAD_CODES_LOOKUP_RPC_NAME = "VBECS WORKLOAD CODES LOOKUP";
	    private static readonly Timeout TIMEOUT = new Timeout(120000);

	    #endregion

		#region Internal Methods

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves Hcpcs Cache
		/// </summary>
		internal static DataSet GetHcpcsCache()
		{
			return StoredProcedure.GetData(STOREDPROC.GetHcpcsCache.StoredProcName);
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves Hospital Location Cache
		/// </summary>
		internal static DataSet GetHospitalLocationCache(string locationDivision)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetHospitalLocationCache.locationdivision, SqlDbType.Char)
			};
			//
			prms[0].Value = locationDivision;
			//
			return StoredProcedure.GetData(STOREDPROC.GetHospitalLocationCache.StoredProcName, prms);
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves Workload Cache; includes Workload and Cpt Codes
		/// </summary>
		internal static DataSet GetWorkloadAndCptCache()
		{
			var dsWorkloadCodes = StoredProcedure.GetData(STOREDPROC.GetWorkloadCache.StoredProcName);
			//
			if (dsWorkloadCodes != null && dsWorkloadCodes.Tables.Count == 2 && dsWorkloadCodes.Tables[0].Rows.Count > 0 && dsWorkloadCodes.Tables[1].Rows.Count > 0)
			{
				dsWorkloadCodes.Tables[0].TableName = WORKLOAD_TABLE_NAMES.Code;
				dsWorkloadCodes.Tables[1].TableName = WORKLOAD_TABLE_NAMES.CPTCode;
			}
			//
			// This DR seems to get automatically created when you call
			// DataTransformUtility.LoadXmlStringIntoDataSet 
			// (see WorkloadCodes.GetWorkloadCodeData())
			// so, we have to create it for the cached data too
			var code_CPTCode = new DataRelation 
			(
				"Code_CPTCode", 
				dsWorkloadCodes.Tables[WORKLOAD_TABLE_NAMES.Code].Columns["Code_Id"], 
				dsWorkloadCodes.Tables[WORKLOAD_TABLE_NAMES.CPTCode].Columns["Code_Id"]
			);
			//
			dsWorkloadCodes.Relations.Add ( code_CPTCode );
			//
			return dsWorkloadCodes;
		}

		#endregion

		#region Private Methods

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Insert Hcpcs into VBECS table
		/// </summary>
		private static void InsertHcpcsCache()
		{
			if (VistALink.SilentCheckStatusReconnectIfNeeded())
			{
                var rpcResult = VistALink.GetRpcBroker().ExecuteRpc(VistALink.CreateRpcRequest(HCPCS_CODES_LOOKUP_RPC_NAME, TIMEOUT));
				//
				var ds = DataTransformUtility.LoadXmlStringIntoDataSet(rpcResult);
				//
				if (ds != null && ds.Tables.Count == 1 && ds.Tables[0] != null && ds.Tables[0].Rows.Count > 0)
				{
					var storedProcedures = new ArrayList();
					var dataTables = new ArrayList();
					//
					// Clear existing cache so we can insert the latest from VistA
					storedProcedures.Add(STOREDPROC.ClearHcpcsCache.StoredProcName);
					dataTables.Add(new DataTable());
					//
					storedProcedures.Add(STOREDPROC.InsertHcpcsCache.StoredProcName);
					dataTables.Add(ds.Tables[0]);
					//
					new StoredProcedure().SimpleTransactionalGetValue(storedProcedures, dataTables);
				}
			}
		}
		
		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Insert Hospital Location into VBECS table
		/// </summary>
		private static void InsertHospitalLocationCache()
		{
			if (VistALink.SilentCheckStatusReconnectIfNeeded())
			{
				var storedProcedures = new ArrayList();
				var dataTables = new ArrayList();
				//
				// Clear existing cache so we can insert the lastest from VistA
				storedProcedures.Add(STOREDPROC.ClearHospitalLocationCache.StoredProcName);
				dataTables.Add(new DataTable());
				//
				var dtDivisions = Division.GetDivisions(true);
				//
				for (var idx = 0; idx < dtDivisions.Rows.Count; idx++)
				{
					var divisionCode = dtDivisions.Rows[idx][TABLES.VamcDivision.DivisionCode ].ToString();
					//
                    var request = VistALink.CreateRpcRequest(HOSPITAL_LOCATION_LOOKUP_RPC_NAME, TIMEOUT);
					//
					request.Parameters.Add( new RpcStringParameter(1, divisionCode.Trim()) );
					//
					var rpcResult = VistALink.GetRpcBroker().ExecuteRpc(request);
					//
					var ds = DataTransformUtility.LoadXmlStringIntoDataSet(rpcResult);
					//
					if (ds != null && ds.Tables.Count == 1 && ds.Tables[0] != null && ds.Tables[0].Rows.Count > 0)
					{
						storedProcedures.Add(STOREDPROC.InsertHospitalLocationCache.StoredProcName);
						dataTables.Add(AddColumnAppendData(ds.Tables[0], "LocationDivision", divisionCode));
					}
				}
				//
				if (storedProcedures.Count > 1 && dataTables.Count > 1)
				{
					new StoredProcedure().SimpleTransactionalGetValue(storedProcedures, dataTables);
				}
			}
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Insert Workload And Cpt into VBECS table
		/// </summary>
		private static void InsertWorkloadAndCptCache()
		{
			if (VistALink.SilentCheckStatusReconnectIfNeeded())
			{
                var rpcResult = VistALink.GetRpcBroker().ExecuteRpc(VistALink.CreateRpcRequest(WORKLOAD_CODES_LOOKUP_RPC_NAME, TIMEOUT));
				//
				var ds = DataTransformUtility.LoadXmlStringIntoDataSet(rpcResult);
				//
				if (ds != null && ds.Tables.Count == 2 && ds.Tables[0] != null && ds.Tables[0].Rows.Count > 0 && ds.Tables[1] != null && ds.Tables[1].Rows.Count > 0)
				{
					var storedProcedures = new ArrayList();
					var dataTables = new ArrayList();
					//
					// Clear existing cache so we can insert the lastest from VistA
					storedProcedures.Add(STOREDPROC.ClearCptCache.StoredProcName);
					dataTables.Add(new DataTable());
					//
					storedProcedures.Add(STOREDPROC.ClearWorkloadCache.StoredProcName);
					dataTables.Add(new DataTable());
					//
					storedProcedures.Add(STOREDPROC.InsertWorkloadCache.StoredProcName);
					dataTables.Add(ds.Tables[0]);
					//
					storedProcedures.Add(STOREDPROC.InsertCptCache.StoredProcName);
					dataTables.Add(ds.Tables[1]);
					//
					new StoredProcedure().SimpleTransactionalGetValue(storedProcedures, dataTables);
				}
			}
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Add column data to the input DataTable and return the updated DataTable 
		/// </summary>
		private static DataTable AddColumnAppendData( DataTable dtInput, string columnName, string columnData )
		{
			if( dtInput == null )
			{
				return null;
			}
			//
			if( !dtInput.Columns.Contains( columnName ) )
			{
				dtInput.Columns.Add( columnName, typeof(string) );
			}
			//
			for (var idx = 0; idx < dtInput.Rows.Count; idx++)
			{
				dtInput.Rows[idx][columnName] = columnData;
			}
			//
			return dtInput;
		}

		#endregion

		#region Public Methods
		
		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8410"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8411"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Check update time on all cached vistA data and update as needed
		/// </summary>
		///<param name="refreshInterval">Update refresh interval in minutes</param>
		public static void UpdateCache(int refreshInterval)
		{
			// used to determine if it's time to refresh cache
			int checkTime;
			//
			var updateCptCache = false;
			var updateHcpsCache = false;
			var updateHospitalLocationCache = false;
			var updateWorkloadCache = false;
			//
		    //
			var currentTime = DateTime.UtcNow;
			//
			var dsUpdateDates = StoredProcedure.GetData(STOREDPROC.GetCacheLastUpdateDates.StoredProcName);
			//
			// *********************************************************************************************************************************************
			// **** Cpt Cache ***
			if (dsUpdateDates != null && dsUpdateDates.Tables.Count == 4 && dsUpdateDates.Tables[0] != null && dsUpdateDates.Tables[0].Rows.Count == 1)
			{
				var lastUpdateDate = Convert.ToDateTime( dsUpdateDates.Tables[0].Rows[0][TABLES.CptCache.LastUpdateDate] );
				checkTime = DateTime.Compare( currentTime, (lastUpdateDate).AddMinutes(refreshInterval) );
				//
				if(checkTime >= 0)
				{
					updateCptCache = true;
				}
			}
			else
			{
				updateCptCache = true;
			}
			//
			// *********************************************************************************************************************************************
			// **** Hcpcs Cache ***
			if (dsUpdateDates != null && dsUpdateDates.Tables.Count == 4 && dsUpdateDates.Tables[1] != null && dsUpdateDates.Tables[1].Rows.Count == 1)
			{
			    //
				var lastUpdateDate = Convert.ToDateTime( dsUpdateDates.Tables[1].Rows[0][TABLES.HcpcsCache.LastUpdateDate] );
				checkTime = DateTime.Compare( currentTime, (lastUpdateDate).AddMinutes(refreshInterval) );
				//
				if(checkTime >= 0)
				{
					updateHcpsCache = true;
				}
			}
			else
			{
				updateHcpsCache = true;
			}
			//
			// *********************************************************************************************************************************************
			// **** Hospital Location Cache ***
			if (dsUpdateDates != null && dsUpdateDates.Tables.Count == 4 && dsUpdateDates.Tables[2] != null && dsUpdateDates.Tables[2].Rows.Count > 0)
			{
				for (var idx = 0; idx < dsUpdateDates.Tables[2].Rows.Count; idx++)
				{
				    //
					var lastUpdateDate = Convert.ToDateTime( dsUpdateDates.Tables[2].Rows[idx][TABLES.HospitalLocationCache.LastUpdateDate] );
					checkTime = DateTime.Compare( currentTime, (lastUpdateDate).AddMinutes(refreshInterval) );
					//
					if(checkTime >= 0)
					{
						updateHospitalLocationCache = true;
						break;
					}
				}
			}
			else
			{
				updateHospitalLocationCache = true;
			}
			//
			// *********************************************************************************************************************************************
			// **** Workload Cache ***
			
			if (dsUpdateDates != null && dsUpdateDates.Tables.Count == 4 && dsUpdateDates.Tables[3] != null && dsUpdateDates.Tables[3].Rows.Count == 1)
			{
			    //
				var lastUpdateDate = Convert.ToDateTime( dsUpdateDates.Tables[3].Rows[0][TABLES.WorkloadCache.LastUpdateDate] );
				checkTime = DateTime.Compare( currentTime, (lastUpdateDate).AddMinutes(refreshInterval) );
				//
				if(checkTime >= 0)
				{
					updateWorkloadCache = true;
				}
			}
			else
			{
				updateWorkloadCache = true;
			}
			//
			// *********************************************************************************************************************************************
			if (updateHcpsCache)
			{
				InsertHcpcsCache();
			}
			//
			if (updateHospitalLocationCache)
			{
				InsertHospitalLocationCache();
			}
			//
			if (updateCptCache || updateWorkloadCache)
			{
				InsertWorkloadAndCptCache();
			}
		}

		#endregion
	}
}
